import React, { useEffect, useState, useRef } from 'react';
import { View, Text, StyleSheet, Pressable, Animated, Easing } from 'react-native';
import AsyncStorage from '@react-native-async-storage/async-storage';

const CompostProgressBar = () => {
  const [progress, setProgress] = useState(0);
  const fillAnim = useRef(new Animated.Value(0)).current;
  const timerRef = useRef(null);

  useEffect(() => {
    const loadProgress = async () => {
      try {
        const stored = await AsyncStorage.getItem('@compost_progress');
        if (stored !== null) setProgress(Number(stored));
      } catch (e) {
        console.error('Failed to load compost progress:', e);
      }
    };
    loadProgress();
  }, []);

  useEffect(() => {
    const interval = setInterval(() => {
      setProgress((prev) => {
        const next = Math.min(prev + 1, 100);
        AsyncStorage.setItem('@compost_progress', String(next));
        return next;
      });
    }, 30000);
    return () => clearInterval(interval);
  }, []);

  const handlePressIn = () => {
    fillAnim.setValue(0);
    Animated.timing(fillAnim, {
      toValue: 1,
      duration: 3000,
      useNativeDriver: false,
      easing: Easing.linear,
    }).start();

    timerRef.current = setTimeout(async () => {
      setProgress(0);
      await AsyncStorage.setItem('@compost_progress', '0');
    }, 3000);
  };

  const handlePressOut = () => {
    clearTimeout(timerRef.current);
    fillAnim.stopAnimation();
    fillAnim.setValue(0);
  };

  const fillStyle = {
    width: fillAnim.interpolate({
      inputRange: [0, 1],
      outputRange: ['0%', '100%'],
    }),
  };

  return (
    <Pressable
      onPressIn={handlePressIn}
      onPressOut={handlePressOut}
      style={styles.card}
    >
      <Text style={styles.cardTitle}>Composting Progress</Text>
      <View style={styles.progressBarContainer}>
        <View
          style={[
            styles.progressBarFill,
            {
              width: `${progress}%`,
              backgroundColor: progress < 100 ? '#5A7D4E' : '#999',
            },
          ]}
        />
        <Animated.View
          style={[styles.resetOverlay, fillStyle]}
        />
      </View>
      <Text style={styles.cardValue}>{progress}%</Text>
      <Text style={styles.holdText}>Hold 3s to reset</Text>
    </Pressable>
  );
};

const styles = StyleSheet.create({
  card: {
    backgroundColor: '#BBBD9A',
    padding: 20,
    borderRadius: 10,
    marginBottom: 15,
    shadowColor: '#000',
    shadowOpacity: 0.1,
    shadowRadius: 5,
    elevation: 2,
    width: '100%',
    marginTop: 20,
  },
  cardTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    marginBottom: 10,
    color: '#333',
  },
  cardValue: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#FFF',
    marginTop: 10,
  },
  holdText: {
    fontSize: 14,
    color: '#333',
    marginTop: 4,
    textAlign: 'center',
  },
  progressBarContainer: {
    height: 25,
    width: '100%',
    backgroundColor: '#eee',
    borderRadius: 12,
    overflow: 'hidden',
    position: 'relative',
  },
  progressBarFill: {
    height: '100%',
    borderRadius: 12,
    position: 'absolute',
    left: 0,
    top: 0,
    zIndex: 1,
  },
  resetOverlay: {
    height: '100%',
    backgroundColor: 'rgba(255, 255, 255, 0.4)',
    position: 'absolute',
    left: 0,
    top: 0,
    zIndex: 2,
  },
});

export default CompostProgressBar;
